import asyncio
import importlib

from pylog.pylogger import PyLogger

from py_pli.pylib import URPCInterfaceType
from py_pli.pylib import URPCFunctions
from py_pli.pylib import VUnits

from virtualunits.vu_node_application import VUNodeApplication

from urpc_enum.error import FirmwareErrorsDic

from urpc.barcodereaderfunctions import BarcodeReaderFunctions
from urpc.corexymoverfunctions import CoreXYMoverFunctions
from urpc.dualmoverfunctions import DualMoverFunctions
from urpc.fancontrolfunctions import FanControlFunctions
from urpc.measurementfunctions import MeasurementFunctions
from urpc.moverfunctions import MoverFunctions
from urpc.nodefunctions import NodeFunctions
from urpc.serialfunctions import SerialFunctions
from urpc.systemcontrolfunctions import SystemControlFunctions
from urpc.temperaturecontrolfunctions import TemperatureControlFunctions

from bodhi.common.node_io import *


### Endpoint Utility Functions #########################################################################################

# Node Endpoint ########################################################################################################

node_endpoints = {
    'eef'   : {'id':0x0008, 'delay':5, 'mot_file':'EEF_BDH_Node.mot'},
    'mc6'   : {'id':0x0010, 'delay':1, 'mot_file':'MC6_Node.mot'},
    'fmb'   : {'id':0x0020, 'delay':1, 'mot_file':'FMB_Node.mot'},
}

def get_node_endpoint(name) -> NodeFunctions:
    """ Getter for node endpoints. """
    can_id = node_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.NodeEndpoint)
    return endpoint


def get_node_unit(endpoint) -> VUNodeApplication:
    """ Getter for node virtual unit. """
    can_id = node_endpoints[endpoint]['id']
    for node in VUnits.instance.hal.nodes.values():
        if node.canID == can_id:
            return node
        
    raise Exception(f"No virtual unit configured for node endpoint '{endpoint}' with CAN-ID: 0x{can_id:04X}")


async def start_firmware(node_name):
    """ Start the firmware of a node. Does nothing if the firmware is already running. """
    node = get_node_endpoint(node_name)
    if (await node.GetFirmwareInfo())[0] != 1:
        await node.StartFirmware()
        await asyncio.sleep(node_endpoints[node_name]['delay'])
        if (await node.GetFirmwareInfo())[0] != 1:
            raise Exception(f"Failed to start the firmware.")
        status, error = await node.GetStatus()
        if status != 2:
            raise Exception(f"Firmware failed to initialize with error = 0x{error:04X} : {FirmwareErrorsDic.get(error, 'Unknown Error')}")
    
    return f"start_firmware() done"


async def update_firmware(node_name, mot_file=None):
    """ Update the firmware of a node. """
    if mot_file is None:
        mot_file = node_endpoints[node_name]['mot_file']

    PyLogger.logger.info(f"Updating {node_name.upper()} Firmware '{mot_file}'")
    node_unit = get_node_unit(node_name)
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=1, force=True)

    return f"update_firmware() done"


async def update_bootloader(node_name, mot_file='Bootloader.mot'):
    """ Update the bootloader of a node. """
    PyLogger.logger.info(f"Updating {node_name.upper()} Bootloader '{mot_file}'")
    node_unit = get_node_unit(node_name)
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=0, force=True)

    return f"update_bootloader() done"


async def update_fpga(mot_file='EEF_FPGA.mot'):
    """ Update the FPGA firmware of the EEF board. """
    PyLogger.logger.info(f"Updating FPGA '{mot_file}'")
    node_unit =  get_node_unit('eef')
    firmware_updater = VUnits.instance.hal.firmware_updater
    await firmware_updater.flash_node(node_unit, mot_file, node_unit.Name, device=2, force=True)

    return f"update_fpga() done"


# Mover Endpoint ########################################################################################################

mover_endpoints = {
    'excmc' : {'id':0x0110},  # excitation monochromator (MC6 M1)
    'emsmc' : {'id':0x0111},  # emission monochromator (MC6 M2)
    'excls' : {'id':0x0112},  # excitation light switch (MC6 M3)
    'emsls' : {'id':0x0113},  # emission light switch (MC6 M4)
    'xflfw' : {'id':0x0114},  # xenon flash lamp filter Wheel (MC6 M5)
    'pmtfw' : {'id':0x0115},  # pmt filter wheel (MC6 M6)
}

def get_mover_endpoint(name) -> MoverFunctions:
    """ Getter for mover endpoints. """
    can_id = mover_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.MoverEndpoint)
    return endpoint


# Dual Mover Endpoint ######################################################################################################

dual_mover_endpoints = {
    'st'    : {'id':0x0108},    # Scan Table
    'umh'   : {'id':0x010A},    # Upper Measurement Head
}

def get_dual_mover_endpoint(name='st') -> DualMoverFunctions:
    """ Getter for dual mover endpoints. """
    can_id = dual_mover_endpoints[name]['id']
    endpoint =  URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.DualMoverEndpoint)
    return endpoint


# Measurement Endpoint #################################################################################################

measurement_endpoints = {
    'meas'  : {'id':0x010C},
} 

def get_measurement_endpoint(name='meas') -> MeasurementFunctions:
    """ Getter for measurement endpoints. The name is optional, since we only have one endpoint. """
    can_id = measurement_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.MeasurementEndpoint)
    return endpoint


# Serial Endpoint ######################################################################################################

serial_endpoints = {
    'trf'   : {'id':0x010E, 'channel':0},
}

def get_serial_endpoint(name) -> SerialFunctions:
    """ Getter for serial endpoints. """
    can_id = serial_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.SerialEndpoint)
    return endpoint


# Barcode Reader Endpoint ##############################################################################################

barcode_reader_endpoints = {
    'bcr1'  : {'id':0x010D, 'channel':0},
    'bcr2'  : {'id':0x010F, 'channel':0},
    'bcr3'  : {'id':0x010F, 'channel':1},
    'bcr4'  : {'id':0x010F, 'channel':2},
    'bcr5'  : {'id':0x010F, 'channel':3},
}

def get_barcode_reader_endpoint(name) -> BarcodeReaderFunctions:
    """ Getter for barcode reader endpoints. """
    can_id = barcode_reader_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.BarcodeReaderEndpoint)
    return endpoint


# System Control Endpoint ##############################################################################################

system_control_endpoints = {
    'sys'   : {'id':0x0120},
}

def get_system_control_endpoint(name='sys') -> SystemControlFunctions:
    """ Getter for system control endpoints. The name is optional, since we only have one endpoint. """
    can_id = system_control_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.SystemControlEndpoint)
    return endpoint


# Fan Control Endpoint #################################################################################################

fan_control_endpoints = {
    'eef_fan'   : {'id':0x020A},
    'fmb_fan'   : {'id':0x0121},
}

def get_fan_control_endpoint(name='fmb_fan') -> FanControlFunctions:
    """ Getter for fan control endpoints. """
    can_id = fan_control_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.FanControlEndpoint)
    return endpoint


# Temperature Control Endpoint #########################################################################################

temperature_control_endpoints = {
    'eef_tc'    : {'id':0x0209},
    'fmb_tc'    : {'id':0x0122},
}

def get_temperature_control_endpoint(name) -> TemperatureControlFunctions:
    """ Getter for temperature control endpoints. """
    can_id = temperature_control_endpoints[name]['id']
    endpoint = URPCFunctions.instance.endPointCreator.createEndPoint(can_id, URPCInterfaceType.TemperatureControlEndpoint)
    return endpoint


# Endpoint Creator #####################################################################################################

def create_endpoint(can_id, endpoint_class_name):
    """ Create endpoints that are not supported by the PyRunner yet. """
    endpoint_creator = URPCFunctions.instance.endPointCreator

    endpoint_module = importlib.import_module('urpc.' + endpoint_class_name.lower())
    endpoint_class = getattr(endpoint_module, endpoint_class_name)

    endpoint = None
    
    if can_id in endpoint_creator.urpcFunctions.endPointsDic:
        PyLogger.logger.debug(f"Endpoint already exist for CAN ID: {can_id}")
        endpoint = endpoint_creator.urpcFunctions.endPointsDic[can_id]

    if not isinstance(endpoint, endpoint_class):
        PyLogger.logger.debug(f"New Endpoint created for CAN ID: {can_id}")
        endpoint = endpoint_class(can_id, endpoint_creator.transmitterQueueParent, endpoint_creator.receptionQueueChild, endpoint_creator.iso15765xSend, endpoint_creator.event_loop)

        endpoint_creator.creationEvent.clear()
        endpoint_creator.createEndpointQueueParent.send(can_id)
        endpoint_creator.creationEvent.wait()
        endpoint_creator.creationEvent.clear()

        endpoint_creator.urpcFunctions.endPointsDic[can_id] = endpoint

    return endpoint


########################################################################################################################
